/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: User_win32.c $:

  Description:
    USER implemented functions called by the netXTransport Toolkit

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2013-02-23  initial version

**************************************************************************************/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include "netXTransport.h"
#include "HilTransportLayer.h"
#include "ConnectorAPI.h"
#include "OS_Dependent.h"

typedef struct NXT_LOG_FILE_Ttag
{
  void* hnetXTransport;
  FILE* hFile;
} NXT_LOG_FILE_T;

/*****************************************************************************/
/*! Print a trace message from netXtransport toolkit
*     \param ptDevInstance  Device instance the trace is coming from
*     \param ulTraceLevel   see TRACE_LVL_XXX defines
*     \param szFormat       printf style format string
*     \param ...            printf arguments                                 */
/*****************************************************************************/
void USER_Trace(NETX_TRANSPORT_DATA_T* ptnetXTransportInst, uint32_t ulTraceLevel, char* szFormat, ...)
{
  va_list         vaformat;
  char            szVarstring[NXT_MAX_PATH];
  char            szBuffer[NXT_MAX_PATH];
  SYSTEMTIME      SystemTime;
  NXT_LOG_FILE_T* ptLogFile;

  ptLogFile = (NXT_LOG_FILE_T*)ptnetXTransportInst->pvLogFile;

  UNREFERENCED_PARAMETER(ulTraceLevel);

  GetLocalTime( &SystemTime);
  sprintf_s( szBuffer, NXT_MAX_PATH,"%.2d.%.2d.%.4d %.2d:%.2d:%.2d.%.3d: ",
    SystemTime.wDay,
    SystemTime.wMonth,
    SystemTime.wYear,
    SystemTime.wHour,
    SystemTime.wMinute,
    SystemTime.wSecond,
    SystemTime.wMilliseconds);

  va_start( vaformat, szFormat);

  vsnprintf_s( szVarstring, NXT_MAX_PATH, NXT_MAX_PATH-strlen(szBuffer),szFormat, vaformat);
  va_end( vaformat);

  strcat_s(szBuffer, NXT_MAX_PATH, szVarstring);

  /* check if log file exists */
  if (ptLogFile && ptLogFile->hFile)
  {
    fprintf(ptLogFile->hFile, "%s\n",szBuffer);
  } else
  {
    printf("%s\n",szBuffer);
  }
}

/*****************************************************************************/
/*! Initialize trace
*     \param ptnetXTransportInst Pointer to netXTransport instance           */
/*****************************************************************************/
void USER_TraceInitialize(NETX_TRANSPORT_DATA_T* ptnetXTransportInst)
{
  NXT_LOG_FILE_T* ptLogFile;

  ptLogFile = OS_Memalloc(sizeof(NXT_LOG_FILE_T));
  OS_Memset(ptLogFile, 0, sizeof(NXT_LOG_FILE_T));

  ptnetXTransportInst->pvLogFile = (void*)ptLogFile;

  /* check if log file exists */
  if (ptLogFile)
  {
    int iRet;
    if (0 != ( iRet = fopen_s( &ptLogFile->hFile, "netXTransport.log","w+")))
    {
      printf("Error while creating log file. Print Debug-Messages to console!\n");
    }
  }
  /* log file header */
  USER_Trace(ptnetXTransportInst, 0xFF, "----- netXTransport Log started ---------------------\n");
  USER_Trace(ptnetXTransportInst, 0xFF, "netXTransport Toolkit-Version: %d.%d.%d.%d\n", NXT_VERSION_MAJOR, NXT_VERSION_MINOR, NXT_VERSION_REV, NXT_VERSION_BUILD);
}

/*****************************************************************************/
/*! De-initialize trace
*     \param ptnetXTransportInst Pointer to netXTransport instance           */
/*****************************************************************************/
void USER_TraceDeInitialize(NETX_TRANSPORT_DATA_T* ptnetXTransportInst)
{
  void* pvLogFile = ptnetXTransportInst->pvLogFile;

  if (pvLogFile)
  {
    NXT_LOG_FILE_T* ptLogFile = pvLogFile;
    if (ptLogFile->hFile)
    {
      fclose(ptLogFile->hFile);
    }
    OS_Memfree(ptLogFile);
    ptnetXTransportInst->pvLogFile = NULL;
  }
}

/*****************************************************************************/
/*! Function retrieves connector specific timeout
*     \param ptConnector  Pointer to connector
*     \param ptConnector  Pointer to returned timeout
/*****************************************************************************/
void USER_GetConnectorTimeout( PNETX_CONNECTOR_T ptConnector, uint32_t* pulTimeout)
{
  if (pulTimeout == NULL)
    return;

  UNREFERENCED_PARAMETER( ptConnector);

  /* currently set to default timeout */
  *pulTimeout = 100;
}

/*****************************************************************************/
/*! User defined send thread function (starts HilTransport Send-Handler)
*     \param lpParameter Pointer to win32 dependent thread information structure
*     returns 0                                                              */
/*****************************************************************************/
DWORD WINAPI USERSendThread( LPVOID lpParameter)
{
  WIN32_THREAD_PARAM_T* ptThreadParam = (WIN32_THREAD_PARAM_T*)lpParameter;

  if (NULL != ptThreadParam->pfnThreadFunction)
  {
    ptThreadParam->pfnThreadFunction( ptThreadParam->pvUserParam);
  }
  return 0;
}

/*****************************************************************************/
/*! User defined timeout thread function (starts HilTransport Timeout-Handler)
*     \param lpParameter Pointer to win32 dependent thread information structure
*     returns 0                                                              */
/*****************************************************************************/
DWORD WINAPI USERTimeoutThread( LPVOID lpParameter)
{
  WIN32_THREAD_PARAM_T* ptThreadParam = (WIN32_THREAD_PARAM_T*)lpParameter;

  if (NULL != ptThreadParam->pfnThreadFunction)
  {
    ptThreadParam->pfnThreadFunction( ptThreadParam->pvUserParam);
  }
  return 0;
}

/*****************************************************************************/
/*! Function retrieves OS specific thread parameter
*     \param szThreadName     Name of thread to be started
*     \param ppvThreadParam   Returned pointer to thread parameter
*     \param pulParamSize     Size of buffer pointed by *ppvThreadParam
*     \param pfThreadFunction Pointer to returned timeout parameter info
*     return NXT_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t USER_GetThreadParam( char* szThreadName, void** ppvThreadParam, uint32_t* pulParamSize, PFN_THREAD_FUNCTION pfThreadFunction, void* pvParam)
{
  WIN32_THREAD_PARAM_T* ptThreadParam = NULL;

  if (szThreadName == NULL)
    return -1;
  else if (0 == strncmp(szThreadName, "HilTransportSendThread", strlen(szThreadName)))
  {
    ptThreadParam = (WIN32_THREAD_PARAM_T*)OS_Memalloc(sizeof(WIN32_THREAD_PARAM_T));

    ptThreadParam->lptSecAtt         = NULL;
    ptThreadParam->tStackSize        = 0;
    ptThreadParam->lpStartAddr       = USERSendThread;
    ptThreadParam->pvParam           = ptThreadParam;
    ptThreadParam->dwFlags           = 0;
    ptThreadParam->lpID              = 0;
    ptThreadParam->pfnThreadFunction = pfThreadFunction;
    ptThreadParam->pvUserParam       = pvParam;

  } else if (0 == strncmp(szThreadName, "HilTransportTimeoutThread", strlen(szThreadName)))
  {
    ptThreadParam = (WIN32_THREAD_PARAM_T*)OS_Memalloc(sizeof(WIN32_THREAD_PARAM_T));

    ptThreadParam->lptSecAtt         = NULL;
    ptThreadParam->tStackSize        = 0;
    ptThreadParam->lpStartAddr       = USERTimeoutThread;
    ptThreadParam->pvParam           = ptThreadParam;
    ptThreadParam->dwFlags           = 0;
    ptThreadParam->lpID              = 0;
    ptThreadParam->pfnThreadFunction = pfThreadFunction;
    ptThreadParam->pvUserParam       = pvParam;
  }

  if (ppvThreadParam)
    *ppvThreadParam = ptThreadParam;

  if (pulParamSize)
    *pulParamSize = sizeof(*ptThreadParam);

  return 0;
}

/*****************************************************************************/
/*! Function frees user thread prameter (previously allocated with with USER_GetThreadParam())
*     \param ptConnector  Pointer to connector
*     \param ptConnector  Pointer to returned timeout
/*****************************************************************************/
int32_t USER_ReleaseThreadParam( void* pvThreadParam)
{
  OS_Memfree( pvThreadParam);

  return 0;
}